-------------------------------------------------------------------------------
--
-- $Header: rdbms/admin/dbmsdmbl.sql /main/6 2008/09/28 04:22:10 mmcampos Exp $
--
-- Copyright (c) 2008, Oracle and/or its affiliates. All rights reserved.
--
--    NAME
--    dmblast.sql - Data Mining BLAST type and interface definitions
--
--    DESCRIPTION
--    Header and body for the following NCBI BLAST table functions:
--      BLASTN_MATCH
--      BLASTP_MATCH
--      TBLAST_MATCH
--      BLASTN_ALIGN
--      BLASTP_ALIGN
--      TBLAST_ALIGN
--
--    NOTES
--    . This file has to be encrypted and shipped as .plb, given that
--      - there is no explicit package definition
--      - the type definitions for input and output are specified here.
--      - the ref cursor definitions are specified here.
--
--    . Each procedure/function has a description, listing EXCEPTIONS
--      raised and SCHEMA objects accessed along with some other information.
--      However, EXCEPTIONS and SCHEMA sections include only those exceptions
--      and schema objects, that are thrown or accessed DIRECTLY, not through
--      the call to other procedures/functions.
--
--     MODIFIED  (MM/DD/YY)  COMMENT
--       mmcampos 09/24/08 - Bug-7143878: grant public execute on dmbao,
--                           dmbaos, dmbmo, dmbmos, dmbco, dmbcos
--       mmcampos 05/15/08 - XbranchMerge mmcampos_bug-7027933 from
--                           st_rdbms_11.1.0
--       mmcracke 10/24/05 - Migrate from DMSYS to SYS 
--       shthomas 02/09/05 - Fix the security bug 4000491 
--       shthomas 12/08/04 - Add definition of gap_list object 
--       xbarr    06/25/04 - xbarr_dm_rdbms_migration
--       cbhagwat 06/18/04 - cbhagwat_txn111952
--       shthomas 05/17/04 - Add BLASTN_COMPRESS function
--       shthomas 11/04/03 - Set defaults for subsequence_from and subsequence_to
--       shthomas 10/15/03 - Change substitution matrix to a string
--       mjaganna 04/11/03 - Enable Parallelism
--       mjaganna 04/11/03 - 64 bit OS porting fix
--       mjaganna 01/17/03 - Define prepare for ALIGN TFs
--       mjaganna 01/17/03 - Augment ALIGN output type
--       mjaganna 01/14/03 - Add Prepare functions for MATCH TFs
--       mjaganna 01/14/03 - Add shared context to implementation types
--       mjaganna 12/17/02 - Add rws_ptr parameter to Start functions
--       shthomas 11/25/02 - Change translation type to string
--       mjaganna 09/30/02 - Dont drop and re-create DMSYS
--       ramkrish 07/10/02 - fine-tune numbers to integers - minimize OCINumber
--       ramkrish 07/04/02 - make string indicators numeric
--       ramkrish 06/06/02 - dmbaos and dmbmos type change and unit tests
--       ramkrish 05/28/02 - final 10iR1 signatures
--       ramkrish 05/28/02 - change BLASTN signature to test
--       ramkrish 04/01/02 - change blastn prototype signature
--       ramkrish 03/29/02 - add dmsys user creation (temporary)
--       ramkrish 03/23/02 - fix compilation errors
--       ramkrish 03/21/02 - add Implementation types
--       ramkrish 03/19/02 - ramkrish_blast_tf_template
--       ramkrish 03/19/02 - BLASTN_MATCH prototype
--       ramkrish 03/18/02 - Creation
--
-------------------------------------------------------------------------------
--
--            PRIVATE CONSTANTS, TYPES, FUNCTIONS AND PACKAGES
--
-------------------------------------------------------------------------------

------------
-- CONSTANTS
--

--------
-- TYPES
--
-- NOTES:
--   (1) The types are named in this cryptic manner to enable creation of
--   compactly name C structs for the trusted code. The corr. C structs
--   for these types will be generated by OTT with type indicators. Type
--   names are registered in dmblasti.typ, which will be used by the OTT
--   command line utility to generate the corresponding C structs in the
--   header dmbtyp.h. As the SQL input and output typedefs change and grow
--   over time, the use of OTT will help us keep the C structs and their
--   corresponding indicators in synch.
--
--   For e.g., if we install the types defined here in scott/tiger and
--   run the following command, the file typ.h will automatically be
--   updated with the correct C structs; dmblast.out will contain info
--   about the translation. We can then copy the structs from typ.h into
--   dmbtyp.h (which contains other structs also)
--
--   ott userid=scott/tiger intype=dmblasti.typ outtype=dmblast.out \
--       code=c hfile=typ.h
--
--   Now consider a type named MatchOutput_t. OTT will generate a C struct
--        struct MatchOutput_t {} and an indicator struct
--        struct MatchOutput_Ind_t {}
--   OCCS requires all struct members to have the struct name appended to
--   their names. So a member variable sequence_id would have to be called
--   sequence_id_MatchOutput_t. Pretty soon, these long names will make the
--   code unreadable and coding difficult. The same logic applies to upper/
--   lower case symbols - we will keep them uniformly lower-case.
--
--   However, we should not use mnemonics or short forms for the SQL struct
--   member names because they correspond to actual column names returned
--   by the table function in the server -- ramkrish/031902
--
--   (2) The dmbmnimp Implementation Type is defined with pragma AUTONOMOUS
--   to accomodate possible DML statements during Fetch - we might need
--   this for temporary object creation, or inserts to metadata tables.
--
-------------------------------------------------------------------------------
-- ACHTUNG!! ACHTUNG!! BEGIN- 2B MOVED TO dmcrt.sql AS PART OF SECURITY PROJECT
--
-- Create DMSYS user
-- 
--DROP USER dmsys CASCADE;
--CREATE USER dmsys IDENTIFIED BY dmsys
-- DEFAULT TABLESPACE SYSAUX
--  TEMPORARY TABLESPACE TEMP
-- QUOTA UNLIMITED ON SYSAUX
--  ;
--GRANT ALL PRIVILEGES TO dmsys WITH ADMIN OPTION;

-- Run this script as SYS
ALTER SESSION SET CURRENT_SCHEMA = "SYS";

-- ACHTUNG!! ACHTUNG!! END- 2B MOVED TO dmcrt.sql AS PART OF SECURITY PROJECT
-------------------------------------------------------------------------------

-- define BLAST Library
--
CREATE LIBRARY DMBLAST_LIB TRUSTED AS STATIC;
/

-----------------------------
-- TYPES FOR BLAST GAP OUTPUT
-----------------------------

-- DM Blast Gap Output (dmbgo)
--
CREATE OR REPLACE TYPE dmbgo AS OBJECT (
  gap_start  INTEGER,
  gap_type   INTEGER,
  gap_length INTEGER
)
/

-- DM Blast Gap Output Set (dmbgos)
--
CREATE OR REPLACE TYPE dmbgos AS TABLE OF dmbgo
/

GRANT EXECUTE ON dmbgo TO PUBLIC
/
GRANT EXECUTE ON dmbgos TO PUBLIC
/
CREATE OR REPLACE PUBLIC SYNONYM dmbgos FOR dmbgos
/

------------------------
-- TYPES FOR BLAST MATCH
------------------------

-- DM Blast Match Output (dmbmo)
--
CREATE OR REPLACE TYPE dmbmo AS OBJECT (
  t_seq_id VARCHAR2(4000),
  score    INTEGER,
  expect   NUMBER
)
/
GRANT EXECUTE ON dmbmo TO PUBLIC
/

-- DM Blast Match Output Set (dmbmos)
--
CREATE OR REPLACE TYPE dmbmos AS TABLE OF dmbmo
/
GRANT EXECUTE ON dmbmos TO PUBLIC
/

------------------------
-- TYPES FOR BLAST ALIGN
------------------------

-- DM Blast Align Output (dmbao) 
--
CREATE OR REPLACE TYPE dmbao AS OBJECT (
  t_seq_id         VARCHAR2(4000),
  pct_identity     NUMBER,
  alignment_length INTEGER,
  positives        INTEGER,
  mismatches       INTEGER,
  gap_openings     INTEGER,
  gap_list         dmbgos,
  q_seq_start      INTEGER,
  q_seq_end        INTEGER,
  q_frame          INTEGER,
  t_seq_start      INTEGER,
  t_seq_end        INTEGER,
  t_frame          INTEGER,
  score            INTEGER,
  expect           NUMBER  
)
/
GRANT EXECUTE ON dmbao TO PUBLIC
/

-- DM Blast Align Output Set (dmbaos)
--
CREATE OR REPLACE TYPE dmbaos AS TABLE OF dmbao
/
GRANT EXECUTE ON dmbaos TO PUBLIC
/

---------------------------
-- TYPES FOR BLAST COMPRESS
---------------------------

-- DM Blast Compress Output (dmbco)
--
CREATE OR REPLACE TYPE dmbco AS OBJECT (
  t_seq_id VARCHAR2(4000),
  t_seq_data BLOB
)
/
GRANT EXECUTE ON dmbco TO PUBLIC
/

-- DM Blast Compress Output Set (dmbcos)
--
CREATE OR REPLACE TYPE dmbcos AS TABLE OF dmbco
/
GRANT EXECUTE ON dmbcos TO PUBLIC
/
SHOW ERRORS;

------------
-- FUNCTIONS
--

-----------
-- PACKAGES
--

-- Cursor package - just to group all cursors together in a package
--
CREATE OR REPLACE PACKAGE blast_cur AS
  TYPE refcur_t IS REF CURSOR;
  TYPE seqdbrec_t IS RECORD (
    sequence_id VARCHAR2(4000),
    sequence_db CLOB
  );
  TYPE seqdbcur_t IS REF CURSOR RETURN seqdbrec_t;
END blast_cur;
/
SHOW ERRORS;

-------------------------------------------------------------------------------
--
--              T A B L E    F U N C T I O N    D E F I N I T I O N S
--
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
--                             BLAST N MATCH
-------------------------------------------------------------------------------

-- DM BlastN Match IMPlementation type definition (dmbmnimp)
--
CREATE OR REPLACE TYPE dmbmnimp AUTHID CURRENT_USER AS OBJECT (
  key RAW(4),
  shared_ctx RAW(8),
  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbmnimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 5,         
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 2,
                    mismatch_cost         IN  PLS_INTEGER  DEFAULT -3,
                    match_reward          IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 11,
                    xdropoff              IN  PLS_INTEGER  DEFAULT 30,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 50
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbmnimp,
                    nrows            IN     NUMBER,
                    outset           OUT    dmbmos
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbmnimp
                                 ) RETURN NUMBER,  
  STATIC FUNCTION ODCITablePrepare(
                    sctx                  IN OUT dmbmnimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN RAW,
                    heap_ptr              IN RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 5,         
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 2,
                    mismatch_cost         IN  PLS_INTEGER  DEFAULT -3,
                    match_reward          IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 11,
                    xdropoff              IN  PLS_INTEGER  DEFAULT 30,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 50
                   ) return PLS_INTEGER,
  STATIC FUNCTION dmbmnStartStub (
                    sctx                  IN OUT dmbmnimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    mismatch_cost         IN  PLS_INTEGER,
                    match_reward          IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    xdropoff              IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbmnFetchStub (
                    self             IN OUT dmbmnimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbmos
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbmnCloseStub (
                    self             IN  dmbmnimp
                                 ) RETURN NUMBER,
  STATIC FUNCTION dmbmnPrepareStub(
                    sctx                  IN OUT dmbmnimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN RAW,
                    heap_ptr              IN RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    mismatch_cost         IN  PLS_INTEGER,
                    match_reward          IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    xdropoff              IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                   ) return  PLS_INTEGER
);
/
SHOW ERRORS;

-- DM BlastN Match IMPlementation type body (dmbmnimp)
--
CREATE OR REPLACE TYPE BODY dmbmnimp IS

  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbmnimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    mismatch_cost         IN  PLS_INTEGER,
                    match_reward          IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    xdropoff              IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbmnimp.dmbmnStartStub (
             sctx,
             rws_ptr,
             query_sequence,             
             seqdb_cursor,
             subsequence_from,
             subsequence_to,
             filter_low_complexity,
             mask_lower_case,
             expect_value,
             open_gap_cost,
             extend_gap_cost,
             mismatch_cost,
             match_reward,
             word_size,
             xdropoff,
             final_x_dropoff);
  END;
  
  STATIC FUNCTION dmbmnStartStub (
                    sctx                  IN OUT dmbmnimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    mismatch_cost         IN  PLS_INTEGER,
                    match_reward          IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    xdropoff              IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_n_start"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      mismatch_cost SB2,
      mismatch_cost INDICATOR,
      match_reward UB2,
      match_reward INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      xdropoff UB2,
      xdropoff INDICATOR,    
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,
      rws_ptr OCIRaw,
      RETURN OCINumber
    );

  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbmnimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbmos
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbmnFetchStub (nrows, outset);
  END;

  MEMBER FUNCTION dmbmnFetchStub (
                    self             IN OUT dmbmnimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbmos
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_n_fetch"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      nrows,
      nrows INDICATOR,
      outset,
      outset INDICATOR,
      RETURN OCINumber
    );
                                    
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbmnimp
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbmnCloseStub();
  END;

  MEMBER FUNCTION dmbmnCloseStub (
                    self             IN  dmbmnimp
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_n_close"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      RETURN OCINumber
    );
  
  STATIC FUNCTION ODCITablePrepare (
                    sctx                  IN OUT dmbmnimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN RAW,
                    heap_ptr              IN RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    mismatch_cost         IN  PLS_INTEGER,
                    match_reward          IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    xdropoff              IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) return PLS_INTEGER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
     RETURN dmbmnimp.dmbmnPrepareStub (
       sctx,
       tf_info,
       qkn_ptr,
       heap_ptr,
       query_sequence,
       seqdb_cursor,
       subsequence_from,
       subsequence_to,
       filter_low_complexity,
       mask_lower_case,
       expect_value,
       open_gap_cost,
       extend_gap_cost,
       mismatch_cost,
       match_reward,
       word_size,
       xdropoff,
       final_x_dropoff);
  END;
  
  STATIC FUNCTION dmbmnPrepareStub (
                    sctx                  IN OUT dmbmnimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    mismatch_cost         IN  PLS_INTEGER,
                    match_reward          IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    xdropoff              IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) return PLS_INTEGER  
    AS language C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_n_prepare"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      tf_info,
      tf_info INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      mismatch_cost SB2,
      mismatch_cost INDICATOR,
      match_reward UB2,
      match_reward INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      xdropoff UB2,
      xdropoff INDICATOR,    
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,    
      qkn_ptr OCIRaw,
      heap_ptr OCIRaw,
      return INT
    );
END;
/
SHOW ERRORS;

-- GRANT EXECUTE ON dmbmnimp TO PUBLIC WITH GRANT OPTION;

-- BLASTN_MATCH Table Function
--
CREATE OR REPLACE FUNCTION blastn_match (
  query_sequence        IN CLOB,  
--  seqdb_cursor             blast_cur.seqdbcur_t,
  seqdb_cursor             SYS_REFCURSOR,
  subsequence_from      IN INTEGER  DEFAULT 1,
  subsequence_to        IN INTEGER  DEFAULT -1,
  filter_low_complexity IN INTEGER  DEFAULT 0,
  mask_lower_case       IN INTEGER  DEFAULT 0,
  expect_value          IN NUMBER   DEFAULT 10,
  open_gap_cost         IN INTEGER  DEFAULT 5,         
  extend_gap_cost       IN INTEGER  DEFAULT 2,
  mismatch_cost         IN INTEGER  DEFAULT -3,
  match_reward          IN INTEGER  DEFAULT 1,
  word_size             IN INTEGER  DEFAULT 11,
  xdropoff              IN INTEGER  DEFAULT 30,
  final_x_dropoff       IN INTEGER  DEFAULT 50
)
RETURN dmbmos AUTHID CURRENT_USER
  PARALLEL_ENABLE(PARTITION seqdb_cursor BY ANY) 
  PIPELINED USING dmbmnimp;
/
SHOW ERRORS;

GRANT EXECUTE ON blastn_match TO PUBLIC;

CREATE OR REPLACE PUBLIC SYNONYM blastn_match FOR blastn_match;

-------------------------------------------------------------------------------
--                             BLAST P MATCH
-------------------------------------------------------------------------------

-- DM BlastP Match IMPlementation type definition (dmbmpimp)
--
CREATE OR REPLACE TYPE dmbmpimp AUTHID CURRENT_USER AS OBJECT (
  key RAW(4),
  shared_ctx RAW(8),
  STATIC FUNCTION ODCITablePrepare (
                    sctx                  IN OUT dmbmpimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    sub_matrix            IN  STRING  DEFAULT 'BLOSUM62',
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 11,
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 3,
                    x_dropoff             IN  PLS_INTEGER  DEFAULT 15,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 25
                  ) RETURN PLS_INTEGER,
  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbmpimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    sub_matrix            IN  STRING  DEFAULT 'BLOSUM62',
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 11,
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 3,
                    x_dropoff             IN  PLS_INTEGER  DEFAULT 15,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 25
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbmpimp,
                    nrows            IN     NUMBER,
                    outset           OUT    dmbmos
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbmpimp
                                 ) RETURN NUMBER,
  STATIC FUNCTION dmbmpPrepareStub (
                    sctx                  IN OUT dmbmpimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER,
  STATIC FUNCTION dmbmpStartStub (
                    sctx                  IN OUT dmbmpimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbmpFetchStub (
                    self             IN OUT dmbmpimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbmos
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbmpCloseStub (
                    self             IN  dmbmpimp
                                 ) RETURN NUMBER
);
/
SHOW ERRORS;

-- DM BlastP Match IMPlementation type body (dmbmpimp)
--
CREATE OR REPLACE TYPE BODY dmbmpimp IS
   
  STATIC FUNCTION ODCITablePrepare (
                    sctx                  IN OUT dmbmpimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbmpimp.dmbmpPrepareStub (
                    sctx,
                    tf_info,
                    qkn_ptr,
                    heap_ptr,
                    query_sequence,
                    seqdb_cursor,
                    subsequence_from,
                    subsequence_to,
                    filter_low_complexity,
                    mask_lower_case,
                    sub_matrix,
                    expect_value,
                    open_gap_cost,
                    extend_gap_cost,
                    word_size,
                    x_dropoff,
                    final_x_dropoff);
  END;
  
  STATIC FUNCTION dmbmpPrepareStub (
                    sctx                  IN OUT dmbmpimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_p_prepare"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      tf_info,
      tf_info INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      sub_matrix,
      sub_matrix INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      x_dropoff UB2,
      x_dropoff INDICATOR,     
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,
      qkn_ptr OCIRaw,
      heap_ptr OCIRaw,
      RETURN INT
    );
   
  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbmpimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbmpimp.dmbmpStartStub (
                    sctx,
                    rws_ptr,
                    query_sequence,
                    seqdb_cursor,
                    subsequence_from,
                    subsequence_to,
                    filter_low_complexity,
                    mask_lower_case,
                    sub_matrix,
                    expect_value,
                    open_gap_cost,
                    extend_gap_cost,
                    word_size,
                    x_dropoff,
                    final_x_dropoff);
  END;
  
  STATIC FUNCTION dmbmpStartStub (
                    sctx                  IN OUT dmbmpimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_p_start"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      sub_matrix,
      sub_matrix INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      x_dropoff UB2,
      x_dropoff INDICATOR,     
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,
      rws_ptr OCIRaw,
      RETURN OCINumber
    );

  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbmpimp,
                    nrows            IN     NUMBER,
                    outset           OUT    dmbmos
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbmpFetchStub (nrows, outset);
  END;

  MEMBER FUNCTION dmbmpFetchStub (
                    self             IN OUT dmbmpimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbmos
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_p_fetch"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      nrows,
      nrows INDICATOR,
      outset,
      outset INDICATOR,
      RETURN OCINumber
    );
                                    
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbmpimp
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbmpCloseStub();
  END;

  MEMBER FUNCTION dmbmpCloseStub (
                    self             IN  dmbmpimp
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_p_close"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      RETURN OCINumber
   );
END;
/
SHOW ERRORS;

-- GRANT EXECUTE ON dmbmpimp TO PUBLIC WITH GRANT OPTION;

-- BLASTP_MATCH Table Function
--
CREATE OR REPLACE FUNCTION blastp_match (
  query_sequence        IN CLOB,  
  seqdb_cursor             blast_cur.seqdbcur_t,
  subsequence_from      IN INTEGER   DEFAULT 1,
  subsequence_to        IN INTEGER   DEFAULT -1,
  filter_low_complexity IN INTEGER   DEFAULT 0,
  mask_lower_case       IN INTEGER   DEFAULT 0,
  sub_matrix            IN STRING    DEFAULT 'BLOSUM62',
  expect_value          IN NUMBER    DEFAULT 10,
  open_gap_cost         IN INTEGER   DEFAULT 11,         
  extend_gap_cost       IN INTEGER   DEFAULT 1,
  word_size             IN INTEGER   DEFAULT 3,
  x_dropoff             IN INTEGER   DEFAULT 15,  
  final_x_dropoff       IN INTEGER   DEFAULT 25
)
RETURN dmbmos AUTHID CURRENT_user
  PARALLEL_ENABLE(PARTITION seqdb_cursor BY ANY)
  PIPELINED USING dmbmpimp;
/
SHOW ERRORS;

GRANT EXECUTE ON blastp_match TO PUBLIC;

CREATE OR REPLACE PUBLIC SYNONYM blastp_match FOR blastp_match;

-------------------------------------------------------------------------------
--                             T BLAST MATCH
-------------------------------------------------------------------------------

-- DM Blast(T) Match IMPlementation type definition (dmbmtimp)
--
CREATE OR REPLACE TYPE dmbmtimp AUTHID CURRENT_USER AS OBJECT (
  key RAW(4),
  shared_ctx RAW(8),
    STATIC FUNCTION ODCITablePrepare (
                    sctx                  IN OUT dmbmtimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    translation_type      IN  STRING  DEFAULT 'blastx',
                    genetic_code          IN  PLS_INTEGER  DEFAULT 1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    sub_matrix            IN  STRING  DEFAULT 'BLOSUM62',
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 11,
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 3,
                    x_dropoff             IN  PLS_INTEGER  DEFAULT 15,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 25
                  ) RETURN PLS_INTEGER,
  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbmtimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    translation_type      IN  STRING  DEFAULT 'blastx',
                    genetic_code          IN  PLS_INTEGER  DEFAULT 1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    sub_matrix            IN  STRING  DEFAULT 'BLOSUM62',
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 11,
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 3,
                    x_dropoff             IN  PLS_INTEGER  DEFAULT 15,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 25
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbmtimp,
                    nrows            IN     NUMBER,
                    outset           OUT    dmbmos
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbmtimp
                  ) RETURN NUMBER,
  STATIC FUNCTION dmbmtPrepareStub (
                    sctx                  IN OUT dmbmtimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER,
  STATIC FUNCTION dmbmtStartStub (
                    sctx                  IN OUT dmbmtimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbmtFetchStub (
                    self             IN OUT dmbmtimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbmos
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbmtCloseStub (
                    self             IN dmbmtimp
                  ) RETURN NUMBER
);
/
SHOW ERRORS;

-- DM TBlast Match IMPlementation type body (dmbmtimp)
--
CREATE OR REPLACE TYPE BODY dmbmtimp IS
   
   STATIC FUNCTION ODCITablePrepare (
                    sctx                  IN OUT dmbmtimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbmtimp.dmbmtPrepareStub (
                    sctx,
                    tf_info,
                    qkn_ptr,
                    heap_ptr,
                    query_sequence,
                    seqdb_cursor,
                    subsequence_from,
                    subsequence_to,
                    translation_type,
                    genetic_code,
                    filter_low_complexity,
                    mask_lower_case,
                    sub_matrix,
                    expect_value,
                    open_gap_cost,
                    extend_gap_cost,
                    word_size,
                    x_dropoff,
                    final_x_dropoff);
  END;
  
  STATIC FUNCTION dmbmtPrepareStub (
                    sctx                  IN OUT dmbmtimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_t_prepare"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      tf_info,
      tf_info INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      translation_type,
      translation_type INDICATOR,
      genetic_code UB1,
      genetic_code INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      sub_matrix,
      sub_matrix INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      x_dropoff UB2,
      x_dropoff INDICATOR,     
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,
      qkn_ptr OCIRaw,
      heap_ptr OCIRaw,
      RETURN INT
    );
   
  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbmtimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbmtimp.dmbmtStartStub (
                    sctx,
                    rws_ptr,
                    query_sequence,
                    seqdb_cursor,
                    subsequence_from,
                    subsequence_to,
                    translation_type,
                    genetic_code,
                    filter_low_complexity,
                    mask_lower_case,
                    sub_matrix,
                    expect_value,
                    open_gap_cost,
                    extend_gap_cost,
                    word_size,
                    x_dropoff,
                    final_x_dropoff);
  END;
  
  STATIC FUNCTION dmbmtStartStub (
                    sctx                  IN OUT dmbmtimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_t_start"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      translation_type,
      translation_type INDICATOR,
      genetic_code UB1,
      genetic_code INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      sub_matrix,
      sub_matrix INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      x_dropoff UB2,
      x_dropoff INDICATOR,     
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,
      rws_ptr OCIRaw,
      RETURN OCINumber
    );

  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbmtimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbmos
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbmtFetchStub (nrows, outset);
  END;

  MEMBER FUNCTION dmbmtFetchStub (
                    self             IN OUT dmbmtimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbmos
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_t_fetch"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      nrows,
      nrows INDICATOR,
      outset,
      outset INDICATOR,
      RETURN OCINumber
    );
                                    
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbmtimp
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbmtCloseStub();
  END;

  MEMBER FUNCTION dmbmtCloseStub (
                    self             IN  dmbmtimp
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbm_t_close"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      RETURN OCINumber
    );
END;
/
SHOW ERRORS;

-- GRANT EXECUTE ON dmbmtimp TO PUBLIC WITH GRANT OPTION;

-- TBLAST_MATCH Table Function
--
CREATE OR REPLACE FUNCTION tblast_match (
  query_sequence        IN CLOB,  
  seqdb_cursor             blast_cur.seqdbcur_t,
  subsequence_from      IN INTEGER   DEFAULT 1,
  subsequence_to        IN INTEGER   DEFAULT -1,
  translation_type      IN STRING    DEFAULT 'blastx',
  genetic_code          IN INTEGER   DEFAULT 1,
  filter_low_complexity IN INTEGER   DEFAULT 0,
  mask_lower_case       IN INTEGER   DEFAULT 0,
  sub_matrix            IN STRING    DEFAULT 'BLOSUM62',
  expect_value          IN NUMBER    DEFAULT 10,
  open_gap_cost         IN INTEGER   DEFAULT 11,         
  extend_gap_cost       IN INTEGER   DEFAULT 1,
  word_size             IN INTEGER   DEFAULT 3,
  x_dropoff             IN INTEGER   DEFAULT 15,  
  final_x_dropoff       IN INTEGER   DEFAULT 25
)
RETURN dmbmos AUTHID CURRENT_user
PARALLEL_ENABLE(PARTITION seqdb_cursor BY ANY)
PIPELINED USING dmbmtimp;
/
SHOW ERRORS;

GRANT EXECUTE ON tblast_match TO PUBLIC;

CREATE OR REPLACE PUBLIC SYNONYM tblast_match FOR tblast_match;

-------------------------------------------------------------------------------
--                             BLAST N ALIGN
-------------------------------------------------------------------------------

-- DM BlastN Align IMPlementation type definition (dmbanimp)
--
CREATE OR REPLACE TYPE dmbanimp AUTHID CURRENT_USER AS OBJECT (
  key RAW(4),
  shared_ctx RAW(8),
  STATIC FUNCTION ODCITablePrepare (
                    sctx                  IN OUT dmbanimp,  
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,  
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 5,
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 2,
                    mismatch_cost         IN  PLS_INTEGER  DEFAULT -3,
                    match_reward          IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 11,
                    xdropoff              IN  PLS_INTEGER  DEFAULT 30,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 50
                  ) RETURN PLS_INTEGER,
  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbanimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,  
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 5,
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 2,
                    mismatch_cost         IN  PLS_INTEGER  DEFAULT -3,
                    match_reward          IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 11,
                    xdropoff              IN  PLS_INTEGER  DEFAULT 30,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 50
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbanimp,
                    nrows            IN     NUMBER,
                    outset           OUT    dmbaos
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbanimp
                                 ) RETURN NUMBER,
  STATIC FUNCTION dmbanPrepareStub (
                    sctx                  IN OUT dmbanimp,
                    tf_info                  SYS.ODCITabFuncInfo,
                    qkn_ptr               IN RAW,
                    heap_ptr              IN RAW,
                    query_sequence        IN CLOB,
                    seqdb_cursor             SYS_REFCURSOR,
                    subsequence_from      IN PLS_INTEGER,
                    subsequence_to        IN PLS_INTEGER,
                    filter_low_complexity IN PLS_INTEGER,
                    mask_lower_case       IN PLS_INTEGER,
                    expect_value          IN NUMBER,
                    open_gap_cost         IN PLS_INTEGER,
                    extend_gap_cost       IN PLS_INTEGER,
                    mismatch_cost         IN PLS_INTEGER,
                    match_reward          IN PLS_INTEGER,
                    word_size             IN PLS_INTEGER,
                    xdropoff              IN PLS_INTEGER,
                    final_x_dropoff       IN PLS_INTEGER
                  ) RETURN PLS_INTEGER,
  STATIC FUNCTION dmbanStartStub (
                    sctx                  IN OUT dmbanimp,  
                    rws_ptr               IN  RAW,
                    query_sequence        IN CLOB,  
                    seqdb_cursor             SYS_REFCURSOR,
                    subsequence_from      IN PLS_INTEGER,
                    subsequence_to        IN PLS_INTEGER,
                    filter_low_complexity IN PLS_INTEGER,
                    mask_lower_case       IN PLS_INTEGER,
                    expect_value          IN NUMBER,
                    open_gap_cost         IN PLS_INTEGER,
                    extend_gap_cost       IN PLS_INTEGER,
                    mismatch_cost         IN PLS_INTEGER,
                    match_reward          IN PLS_INTEGER,
                    word_size             IN PLS_INTEGER,
                    xdropoff              IN PLS_INTEGER,
                    final_x_dropoff       IN PLS_INTEGER
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbanFetchStub (
                    self             IN OUT dmbanimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbaos
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbanCloseStub (
                    self             IN dmbanimp
                  ) RETURN NUMBER
);
/
SHOW ERRORS;

-- DM BlastN Match Align IMPlementation type body (dmbanimp)
--
CREATE OR REPLACE TYPE BODY dmbanimp IS
   
   STATIC FUNCTION ODCITablePrepare (
                    sctx                  IN OUT dmbanimp,  
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    mismatch_cost         IN  PLS_INTEGER,
                    match_reward          IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    xdropoff              IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbanimp.dmbanPrepareStub (
                    sctx,
                    tf_info,
                    qkn_ptr,
                    heap_ptr,
                    query_sequence,
                    seqdb_cursor,
                    subsequence_from,
                    subsequence_to,
                    filter_low_complexity,
                    mask_lower_case,
                    expect_value,
                    open_gap_cost,
                    extend_gap_cost,
                    mismatch_cost,
                    match_reward,
                    word_size,
                    xdropoff,
                    final_x_dropoff);
  END;

  STATIC FUNCTION dmbanPrepareStub (
                    sctx                  IN OUT dmbanimp,  
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,  
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    mismatch_cost         IN  PLS_INTEGER,
                    match_reward          IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    xdropoff              IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_n_prepare"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      tf_info,
      tf_info INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      mismatch_cost SB2,
      mismatch_cost INDICATOR,
      match_reward UB2,
      match_reward INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      xdropoff UB2,
      xdropoff INDICATOR,    
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,
      qkn_ptr OCIRaw,
      heap_ptr OCIRaw, 
      RETURN INT
    );  
   
  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbanimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,  
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    mismatch_cost         IN  PLS_INTEGER,
                    match_reward          IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    xdropoff              IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbanimp.dmbanStartStub (
                    sctx,
                    rws_ptr,
                    query_sequence,
                    seqdb_cursor,
                    subsequence_from,
                    subsequence_to,
                    filter_low_complexity,
                    mask_lower_case,
                    expect_value,
                    open_gap_cost,
                    extend_gap_cost,
                    mismatch_cost,
                    match_reward,
                    word_size,
                    xdropoff,
                    final_x_dropoff);
  END;

  STATIC FUNCTION dmbanStartStub (
                    sctx                  IN OUT dmbanimp,  
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,  
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    mismatch_cost         IN  PLS_INTEGER,
                    match_reward          IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    xdropoff              IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_n_start"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      mismatch_cost SB2,
      mismatch_cost INDICATOR,
      match_reward UB2,
      match_reward INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      xdropoff UB2,
      xdropoff INDICATOR,    
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,
      rws_ptr OCIRaw, 
      RETURN OCINumber
    );

  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbanimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbaos
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbanFetchStub (nrows, outset);
  END;

  MEMBER FUNCTION dmbanFetchStub (
                    self             IN OUT dmbanimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbaos
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_n_fetch"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      nrows,
      nrows INDICATOR,
      outset,
      outset INDICATOR,
      RETURN OCINumber
    );
                                    
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbanimp
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbanCloseStub();
  END;

  MEMBER FUNCTION dmbanCloseStub (
                    self             IN  dmbanimp
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_n_close"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      RETURN OCINumber
    );
END;
/
SHOW ERRORS;

-- GRANT EXECUTE ON dmbanimp TO PUBLIC WITH GRANT OPTION;

-- BLASTN_ALIGN Table Function
--
CREATE OR REPLACE FUNCTION blastn_align (
  query_sequence        IN CLOB,  
  seqdb_cursor             blast_cur.seqdbcur_t,
  subsequence_from      IN INTEGER   DEFAULT 1,
  subsequence_to        IN INTEGER   DEFAULT -1,
  filter_low_complexity IN INTEGER   DEFAULT 0,
  mask_lower_case       IN INTEGER   DEFAULT 0,
  expect_value          IN NUMBER    DEFAULT 10,
  open_gap_cost         IN INTEGER   DEFAULT 5,         
  extend_gap_cost       IN INTEGER   DEFAULT 2,
  mismatch_cost         IN INTEGER   DEFAULT -3,
  match_reward          IN INTEGER   DEFAULT 1,
  word_size             IN INTEGER   DEFAULT 11,
  xdropoff              IN INTEGER   DEFAULT 30,
  final_x_dropoff       IN INTEGER   DEFAULT 50
)
RETURN dmbaos AUTHID CURRENT_user
PARALLEL_ENABLE(PARTITION seqdb_cursor BY ANY)
PIPELINED USING dmbanimp;
/
SHOW ERRORS;

GRANT EXECUTE ON blastn_align TO PUBLIC;

CREATE OR REPLACE PUBLIC SYNONYM blastn_align FOR blastn_align;

-------------------------------------------------------------------------------
--                             BLAST P ALIGN
-------------------------------------------------------------------------------
-- DM BlastP Match IMPlementation type definition (dmbmpimp)
--
CREATE OR REPLACE TYPE dmbapimp AUTHID CURRENT_USER AS OBJECT (
  key RAW(4),
  shared_ctx RAW(8),
  STATIC FUNCTION ODCITablePrepare (
                    sctx                  IN OUT dmbapimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    sub_matrix            IN  STRING  DEFAULT 'BLOSUM62',
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 11,
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 3,
                    x_dropoff             IN  PLS_INTEGER  DEFAULT 15,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 25
                  ) RETURN PLS_INTEGER,
  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbapimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    sub_matrix            IN  STRING  DEFAULT 'BLOSUM62',
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 11,
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 3,
                    x_dropoff             IN  PLS_INTEGER  DEFAULT 15,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 25
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbapimp,
                    nrows            IN     NUMBER,
                    outset           OUT    dmbaos
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbapimp
                                 ) RETURN NUMBER,
  STATIC FUNCTION dmbapPrepareStub (
                    sctx                  IN OUT dmbapimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER,
  STATIC FUNCTION dmbapStartStub (
                    sctx                  IN OUT dmbapimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbapFetchStub (
                    self             IN OUT dmbapimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbaos
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbapCloseStub (
                    self             IN  dmbapimp
                                 ) RETURN NUMBER
);
/
SHOW ERRORS;

-- DM BlastP Match IMPlementation type body (dmbapimp)
--
CREATE OR REPLACE TYPE BODY dmbapimp IS
   
  STATIC FUNCTION ODCITablePrepare (
                    sctx                  IN OUT dmbapimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbapimp.dmbapPrepareStub (
                    sctx,
                    tf_info,
                    qkn_ptr,
                    heap_ptr,
                    query_sequence,
                    seqdb_cursor,
                    subsequence_from,
                    subsequence_to,
                    filter_low_complexity,
                    mask_lower_case,
                    sub_matrix,
                    expect_value,
                    open_gap_cost,
                    extend_gap_cost,
                    word_size,
                    x_dropoff,
                    final_x_dropoff);
  END;
  
  STATIC FUNCTION dmbapPrepareStub (
                    sctx                  IN OUT dmbapimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_p_prepare"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      tf_info,
      tf_info INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      sub_matrix,
      sub_matrix INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      x_dropoff UB2,
      x_dropoff INDICATOR,     
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,
      qkn_ptr OCIRaw,
      heap_ptr OCIRaw,
      RETURN INT
    );
   
  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbapimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbapimp.dmbapStartStub (
                    sctx,
                    rws_ptr,
                    query_sequence,
                    seqdb_cursor,
                    subsequence_from,
                    subsequence_to,
                    filter_low_complexity,
                    mask_lower_case,
                    sub_matrix,
                    expect_value,
                    open_gap_cost,
                    extend_gap_cost,
                    word_size,
                    x_dropoff,
                    final_x_dropoff);
  END;
  
  STATIC FUNCTION dmbapStartStub (
                    sctx                  IN OUT dmbapimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_p_start"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      sub_matrix,
      sub_matrix INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      x_dropoff UB2,
      x_dropoff INDICATOR,     
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,
      rws_ptr OCIRaw,
      RETURN OCINumber
    );

  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbapimp,
                    nrows            IN     NUMBER,
                    outset           OUT    dmbaos
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbapFetchStub (nrows, outset);
  END;

  MEMBER FUNCTION dmbapFetchStub (
                    self             IN OUT dmbapimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbaos
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_p_fetch"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      nrows,
      nrows INDICATOR,
      outset,
      outset INDICATOR,
      RETURN OCINumber
    );
                                    
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbapimp
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbapCloseStub();
  END;

  MEMBER FUNCTION dmbapCloseStub (
                    self             IN  dmbapimp
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_p_close"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      RETURN OCINumber
   );
END;
/
SHOW ERRORS;

-- GRANT EXECUTE ON dmbapimp TO PUBLIC WITH GRANT OPTION;

-- BLASTP_ALIGN Table Function
--
CREATE OR REPLACE FUNCTION blastp_align (
  query_sequence        IN CLOB,  
  seqdb_cursor             blast_cur.seqdbcur_t,
  subsequence_from      IN INTEGER   DEFAULT 1,
  subsequence_to        IN INTEGER   DEFAULT -1,
  filter_low_complexity IN INTEGER   DEFAULT 0,
  mask_lower_case       IN INTEGER   DEFAULT 0,
  sub_matrix            IN STRING    DEFAULT 'BLOSUM62',
  expect_value          IN NUMBER    DEFAULT 10,
  open_gap_cost         IN INTEGER   DEFAULT 11,         
  extend_gap_cost       IN INTEGER   DEFAULT 1,
  word_size             IN INTEGER   DEFAULT 3,
  x_dropoff             IN INTEGER   DEFAULT 15,  
  final_x_dropoff       IN INTEGER   DEFAULT 25
)
RETURN dmbaos AUTHID CURRENT_USER
PARALLEL_ENABLE(PARTITION seqdb_cursor BY ANY)
PIPELINED USING dmbapimp;
/
SHOW ERRORS;

GRANT EXECUTE ON blastp_align TO PUBLIC;

CREATE OR REPLACE PUBLIC SYNONYM blastp_align FOR blastp_align;

-------------------------------------------------------------------------------
--                             T BLAST ALIGN
-------------------------------------------------------------------------------

-- DM TBlast(T) Align IMPlementation type definition (dmbatimp)
--
CREATE OR REPLACE TYPE dmbatimp AUTHID CURRENT_USER AS OBJECT (
  key RAW(4),
  shared_ctx RAW(8),
  STATIC FUNCTION ODCITablePrepare (
                    sctx                  IN  OUT dmbatimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    translation_type      IN  STRING       DEFAULT 'blastx',
                    genetic_code          IN  PLS_INTEGER  DEFAULT 1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    sub_matrix            IN  STRING  DEFAULT 'BLOSUM62',
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 11,
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 3,
                    x_dropoff             IN  PLS_INTEGER  DEFAULT 15,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 25
                  ) RETURN PLS_INTEGER,
  STATIC FUNCTION ODCITableStart (
                    sctx                  IN  OUT dmbatimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER  DEFAULT 1,
                    subsequence_to        IN  PLS_INTEGER  DEFAULT -1,
                    translation_type      IN  STRING       DEFAULT 'blastx',
                    genetic_code          IN  PLS_INTEGER  DEFAULT 1,
                    filter_low_complexity IN  PLS_INTEGER  DEFAULT 0,
                    mask_lower_case       IN  PLS_INTEGER  DEFAULT 0,
                    sub_matrix            IN  STRING  DEFAULT 'BLOSUM62',
                    expect_value          IN  NUMBER       DEFAULT 10,
                    open_gap_cost         IN  PLS_INTEGER  DEFAULT 11,
                    extend_gap_cost       IN  PLS_INTEGER  DEFAULT 1,
                    word_size             IN  PLS_INTEGER  DEFAULT 3,
                    x_dropoff             IN  PLS_INTEGER  DEFAULT 15,
                    final_x_dropoff       IN  PLS_INTEGER  DEFAULT 25
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbatimp,
                    nrows            IN     NUMBER,
                    outset           OUT    dmbaos
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbatimp
                  ) RETURN NUMBER,
  STATIC FUNCTION dmbatPrepareStub (
                    sctx                  IN  OUT dmbatimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER,
  STATIC FUNCTION dmbatStartStub (
                    sctx                  IN  OUT dmbatimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbatFetchStub (
                    self             IN OUT dmbatimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbaos
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbatCloseStub (
                    self             IN dmbatimp
                  ) RETURN NUMBER
);
/
SHOW ERRORS;

-- DM TBlast(T) Align IMPlementation type body (dmbatimp)
--
CREATE OR REPLACE TYPE BODY dmbatimp IS
   
   STATIC FUNCTION ODCITablePrepare (
                    sctx                  IN  OUT dmbatimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbatimp.dmbatPrepareStub (
                    sctx,
                    tf_info,
                    qkn_ptr,
                    heap_ptr,
                    query_sequence,
                    seqdb_cursor,
                    subsequence_from,
                    subsequence_to,
                    translation_type,
                    genetic_code,
                    filter_low_complexity,
                    mask_lower_case,
                    sub_matrix,
                    expect_value,
                    open_gap_cost,
                    extend_gap_cost,
                    word_size,
                    x_dropoff,
                    final_x_dropoff);
  END;
  
  STATIC FUNCTION dmbatPrepareStub (
                    sctx                  IN OUT dmbatimp,
                    tf_info                   SYS.ODCITabFuncInfo,
                    qkn_ptr               IN  RAW,
                    heap_ptr              IN  RAW,
                    query_sequence        IN  CLOB,
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN PLS_INTEGER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_t_prepare"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      tf_info,
      tf_info INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      translation_type,
      translation_type INDICATOR,
      genetic_code UB1,
      genetic_code INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      sub_matrix,
      sub_matrix INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      x_dropoff UB2,
      x_dropoff INDICATOR,     
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,
      qkn_ptr OCIRaw,
      heap_ptr OCIRaw,
      RETURN INT
    );
   
   STATIC FUNCTION ODCITableStart (
                    sctx                  IN  OUT dmbatimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,            
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbatimp.dmbatStartStub (
                    sctx,
                    rws_ptr,
                    query_sequence,
                    seqdb_cursor,
                    subsequence_from,
                    subsequence_to,
                    translation_type,
                    genetic_code,
                    filter_low_complexity,
                    mask_lower_case,
                    sub_matrix,
                    expect_value,
                    open_gap_cost,
                    extend_gap_cost,
                    word_size,
                    x_dropoff,
                    final_x_dropoff);
  END;
  
  STATIC FUNCTION dmbatStartStub (
                    sctx                  IN OUT dmbatimp,
                    rws_ptr               IN  RAW,
                    query_sequence        IN  CLOB,                    
                    seqdb_cursor              SYS_REFCURSOR,
                    subsequence_from      IN  PLS_INTEGER,
                    subsequence_to        IN  PLS_INTEGER,
                    translation_type      IN  STRING,
                    genetic_code          IN  PLS_INTEGER,
                    filter_low_complexity IN  PLS_INTEGER,
                    mask_lower_case       IN  PLS_INTEGER,
                    sub_matrix            IN  STRING,
                    expect_value          IN  NUMBER,
                    open_gap_cost         IN  PLS_INTEGER,
                    extend_gap_cost       IN  PLS_INTEGER,
                    word_size             IN  PLS_INTEGER,
                    x_dropoff             IN  PLS_INTEGER,
                    final_x_dropoff       IN  PLS_INTEGER
                  ) RETURN NUMBER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_t_start"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      query_sequence OCILobLocator,
      query_sequence INDICATOR sb2,
      seqdb_cursor,
      subsequence_from UB4,
      subsequence_from INDICATOR,
      subsequence_to SB4,
      subsequence_to INDICATOR,
      translation_type,
      translation_type INDICATOR,
      genetic_code UB1,
      genetic_code INDICATOR,
      filter_low_complexity UB1,
      filter_low_complexity INDICATOR,
      mask_lower_case UB1,
      mask_lower_case INDICATOR,
      sub_matrix,
      sub_matrix INDICATOR,
      expect_value,
      expect_value INDICATOR,
      open_gap_cost UB2,
      open_gap_cost INDICATOR,
      extend_gap_cost UB2,
      extend_gap_cost INDICATOR,
      word_size UB1,
      word_size INDICATOR,
      x_dropoff UB2,
      x_dropoff INDICATOR,     
      final_x_dropoff UB2,    
      final_x_dropoff INDICATOR,
      rws_ptr OCIRaw,
      RETURN OCINumber
    );

  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbatimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbaos
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbatFetchStub (nrows, outset);
  END;

  MEMBER FUNCTION dmbatFetchStub (
                    self             IN OUT dmbatimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbaos
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_t_fetch"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      nrows,
      nrows INDICATOR,
      outset,
      outset INDICATOR,
      RETURN OCINumber
    );
                                    
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbatimp
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbatCloseStub();
  END;

  MEMBER FUNCTION dmbatCloseStub (
                    self             IN  dmbatimp
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmba_t_close"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      RETURN OCINumber
    );
END;
/
SHOW ERRORS;

-- GRANT EXECUTE ON dmbatimp TO PUBLIC WITH GRANT OPTION;

-- TBLAST_ALIGN Table Function
--
CREATE OR REPLACE FUNCTION tblast_align (
  query_sequence        IN CLOB,  
  seqdb_cursor             blast_cur.seqdbcur_t,
  subsequence_from      IN INTEGER   DEFAULT 1,
  subsequence_to        IN INTEGER   DEFAULT -1,
  translation_type      IN STRING    DEFAULT 'blastx',
  genetic_code          IN INTEGER   DEFAULT 1,
  filter_low_complexity IN INTEGER   DEFAULT 0,
  mask_lower_case       IN INTEGER   DEFAULT 0,
  sub_matrix            IN STRING    DEFAULT 'BLOSUM62',
  expect_value          IN NUMBER    DEFAULT 10,
  open_gap_cost         IN INTEGER   DEFAULT 11,         
  extend_gap_cost       IN INTEGER   DEFAULT 1,
  word_size             IN INTEGER   DEFAULT 3,
  x_dropoff             IN INTEGER   DEFAULT 15,  
  final_x_dropoff       IN INTEGER   DEFAULT 25
)
RETURN dmbaos AUTHID CURRENT_USER
PARALLEL_ENABLE(PARTITION seqdb_cursor BY ANY)
PIPELINED USING dmbatimp;
/
SHOW ERRORS;

GRANT EXECUTE ON tblast_align TO PUBLIC;

CREATE OR REPLACE PUBLIC SYNONYM tblast_align FOR tblast_align;

-------------------------------------------------------------------------------
--                             BLAST N COMPRESS
-------------------------------------------------------------------------------

-- DM BlastN Compress IMPlementation type definition (dmbcnimp)
--
CREATE OR REPLACE TYPE dmbcnimp AUTHID CURRENT_USER AS OBJECT (
  key RAW(4),
  shared_ctx RAW(8),
  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbcnimp,
                    rws_ptr               IN  RAW,
                    seqdb_cursor              SYS_REFCURSOR
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbcnimp,
                    nrows            IN     NUMBER,
                    outset           OUT    dmbcos
                  ) RETURN NUMBER,
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbcnimp
                                 ) RETURN NUMBER,  
  STATIC FUNCTION dmbcnStartStub (
                    sctx                  IN OUT dmbcnimp,
                    rws_ptr               IN  RAW,
                    seqdb_cursor              SYS_REFCURSOR
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbcnFetchStub (
                    self             IN OUT dmbcnimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbcos
                  ) RETURN NUMBER,
  MEMBER FUNCTION dmbcnCloseStub (
                    self             IN  dmbcnimp
                                 ) RETURN NUMBER
);
/
SHOW ERRORS;

-- DM BlastN Compress IMPlementation type body (dmbcnimp)
--
CREATE OR REPLACE TYPE BODY dmbcnimp IS

  STATIC FUNCTION ODCITableStart (
                    sctx                  IN OUT dmbcnimp,
                    rws_ptr               IN  RAW,
                    seqdb_cursor              SYS_REFCURSOR
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN dmbcnimp.dmbcnStartStub (
             sctx,
             rws_ptr,
             seqdb_cursor);
  END;
  
  STATIC FUNCTION dmbcnStartStub (
                    sctx                  IN OUT dmbcnimp,
                    rws_ptr               IN  RAW,
                    seqdb_cursor              SYS_REFCURSOR
                  ) RETURN NUMBER
  IS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbc_n_start"
    WITH CONTEXT
    PARAMETERS (
      context,
      sctx,
      sctx INDICATOR STRUCT,
      seqdb_cursor,
      rws_ptr OCIRaw,
      RETURN OCINumber
    );

  MEMBER FUNCTION ODCITableFetch (
                    self             IN OUT dmbcnimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbcos
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbcnFetchStub (nrows, outset);
  END;

  MEMBER FUNCTION dmbcnFetchStub (
                    self             IN OUT dmbcnimp,
                    nrows            IN     NUMBER,                    
                    outset           OUT    dmbcos
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbc_n_fetch"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      nrows,
      nrows INDICATOR,
      outset,
      outset INDICATOR,
      RETURN OCINumber
    );
                                    
  MEMBER FUNCTION ODCITableClose (
                    self             IN  dmbcnimp
                  ) RETURN NUMBER
  IS
    PRAGMA AUTONOMOUS_TRANSACTION;
  BEGIN
    RETURN self.dmbcnCloseStub();
  END;

  MEMBER FUNCTION dmbcnCloseStub (
                    self             IN  dmbcnimp
                  ) RETURN NUMBER
  AS
    LANGUAGE C
    LIBRARY DMBLAST_LIB
    NAME "dmbc_n_close"
    WITH CONTEXT
    PARAMETERS (
      context,
      self,
      self INDICATOR STRUCT,
      RETURN OCINumber
    );
  
END;
/
SHOW ERRORS;

-- GRANT EXECUTE ON dmbcnimp TO PUBLIC WITH GRANT OPTION;

-- BLASTN_COMPRESS Table Function
--
CREATE OR REPLACE FUNCTION blastn_compress (
  seqdb_cursor             blast_cur.seqdbcur_t
)
RETURN dmbcos AUTHID CURRENT_USER
  PARALLEL_ENABLE(PARTITION seqdb_cursor BY ANY) 
  PIPELINED USING dmbcnimp;
/
SHOW ERRORS;

GRANT EXECUTE ON blastn_compress TO PUBLIC;

CREATE OR REPLACE PUBLIC SYNONYM blastn_compress FOR blastn_compress;

